<?php
/**
 * Test Email Sending Functionality
 * This script tests if the email system is working correctly by calling the API endpoints
 */

// Test data for sagarnandal95@gmail.com
$test_email = 'sagarnandal95@gmail.com';
$test_name = 'Sagar Nandal';

echo "=== Testing Email Sending Functionality ===\n\n";

// Include the mail system
include_once __DIR__ . '/unified_config.php';

// Test 1: Check if mail system is available
echo "1. Testing Mail System Availability...\n";
if (class_exists('IntegratedMailSystem')) {
    echo "   ✅ IntegratedMailSystem class is available\n";
} else {
    echo "   ❌ IntegratedMailSystem class is NOT available\n";
    exit(1);
}

// Test 2: Initialize mail system
echo "2. Initializing Mail System...\n";
try {
    $mailSystem = getMailSystem();
    echo "   ✅ Mail system initialized successfully\n";
} catch (Exception $e) {
    echo "   ❌ Failed to initialize mail system: " . $e->getMessage() . "\n";
    exit(1);
}

// Test 3: Test sending a simple email using one of the templates
echo "3. Testing Email Sending...\n";
try {
    // Using the deposit_request_submitted template as an example
    $result = $mailSystem->sendDepositRequestSubmitted(
        $test_email,
        $test_name,
        1000.00,     // amount
        'USD',       // currency
        'Bank Transfer', // payment method
        'DEP-TEST-001'   // transaction reference
    );
    
    if ($result['success']) {
        echo "   ✅ Email sent successfully to $test_email\n";
        echo "   Message: " . $result['message'] . "\n";
    } else {
        echo "   ❌ Failed to send email: " . $result['message'] . "\n";
        // Let's check if it's because email automation is disabled
        if (defined('EMAIL_AUTOMATION_ENABLED') && !EMAIL_AUTOMATION_ENABLED) {
            echo "   ℹ️  Email automation is currently disabled\n";
        }
    }
} catch (Exception $e) {
    echo "   ❌ Exception occurred while sending email: " . $e->getMessage() . "\n";
}

// Test 4: Check configuration
echo "\n4. Checking Configuration...\n";
echo "   EMAIL_AUTOMATION_ENABLED: " . (defined('EMAIL_AUTOMATION_ENABLED') ? (EMAIL_AUTOMATION_ENABLED ? 'true' : 'false') : 'not defined') . "\n";
echo "   MAIL_HOST: " . (defined('MAIL_HOST') ? MAIL_HOST : 'not defined') . "\n";
echo "   MAIL_PORT: " . (defined('MAIL_PORT') ? MAIL_PORT : 'not defined') . "\n";
echo "   MAIL_USERNAME: " . (defined('MAIL_USERNAME') ? MAIL_USERNAME : 'not defined') . "\n";

// Test 5: Test all key templates that should be working
echo "\n5. Testing Key Email Templates...\n";
$templates_to_test = [
    'register_verification' => function() use ($mailSystem, $test_email, $test_name) {
        return $mailSystem->sendRegisterVerification($test_email, $test_name, 'TEST-CODE-123');
    },
    'login_alert' => function() use ($mailSystem, $test_email, $test_name) {
        return $mailSystem->sendLoginAlert($test_email, $test_name, '192.168.1.100', 'Test Device');
    },
    'kyc_upload' => function() use ($mailSystem, $test_email, $test_name) {
        return $mailSystem->sendKYCUpload($test_email, $test_name, 'KYC-TEST-001', 'ID Document, Address Proof');
    },
    'withdrawal_request_submitted' => function() use ($mailSystem, $test_email, $test_name) {
        return $mailSystem->sendWithdrawalRequestSubmitted($test_email, $test_name, 500.00, 'USD', 'Bank Transfer', 'WTH-TEST-001');
    }
];

foreach ($templates_to_test as $template_name => $test_function) {
    try {
        $result = $test_function();
        if ($result['success']) {
            echo "   ✅ $template_name: Success\n";
        } else {
            echo "   ⚠️  $template_name: " . $result['message'] . "\n";
        }
    } catch (Exception $e) {
        echo "   ❌ $template_name: Exception - " . $e->getMessage() . "\n";
    }
}

echo "\n=== Test Complete ===\n";
echo "If emails were sent successfully, please check $test_email for the test messages.\n";
?>